/* Time-stamp: <2006-03-28 09:21:19 poser>
 *
 * Program to design a Scrabble-type game given a word list,
 * if lexical frequency is to be used, or a text, if text
 * frequency is to be used.
 *
 * Copyright (C) 1993-2006 William J. Poser.
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or go to the web page:  http://www.gnu.org/licenses/gpl.txt.
 *
 * Author:  Bill Poser
 *	 
 */


#define CHARCNT 256

#include "config.h"
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#ifdef POWERC						/* DOS version */
#include <bios.h>
#include <dos.h>
#include <sys\types.h>
#include <sys\timeb.h>
#endif

#define TRUE 1
#define FALSE 0
#define RND(x) ((int)(x + 0.5))

#define BLANKS 2
#define DEFAULTTILES 100
#define DEFMAXVALUE 10
#define FREQTHRESHOLD 0.001

char progname[]="scrabdes";
char version[]=PACKAGE_VERSION;
char compdate[]= "[" __DATE__ " " __TIME__ "]";

static struct{
   long tokens;
   double frequency;
   int value;
   double ftiles;
   int tiles;
   int expectation;
} cinfo[CHARCNT];

FILE *
OpenFile(char *file,char *mode,char *progname)
{
   FILE *fp;
   extern FILE *fopen();
   static char modestr[7];
   
   if((fp = fopen(file,mode)) != NULL) return fp;
   else{
      if(*mode == 'r') sprintf(modestr,"read");
      else if(*mode == 'w') sprintf(modestr,"write");
      else sprintf(modestr,"append");
      fprintf(stderr,"%s: can't open file %s for %s.\n",progname,file,modestr);
      exit(4);
   }
   /* NOTREACHED */
}

void Usage() {
  fprintf(stderr,"Design a Scrabble-type game given a word list.\n");
  fprintf(stderr,"Output is written on <word list name>.des\n");
  fprintf(stderr,"Usage: %s [options] <input file name>\n",progname);
  fprintf(stderr,"\t-n <tiles desired>\n");
  fprintf(stderr,"\t-m <maximum value>\n");
  fprintf(stderr,"\t-h print help message\n");
  fprintf(stderr,"\t-v print version\n");
  fprintf(stderr,"Report bugs to: billposer@alum.mit.edu\n");
}

void
Version(FILE *fp)   
{
  fprintf(fp,"%s %s\n",progname,version);
  fprintf(fp,gettext("Compiled %s\n"),compdate);
}

void
Copyright(FILE *fp)
{
  fprintf(fp,"Copyright 1993-2006 William J. Poser\n");
  fprintf(fp,"This program is free software; you can redistribute\n\
it and/or modify it under the terms of version 2 of\n\
the GNU General Public License as published by the\n\
Free Software Foundation.\n");
}

main(int ac, char **av)
{
   register int c;
   int opt;
   int i;
   long TotalChars;
   int TilesDesired;
   int TargetTiles;
   int Iterations;
   int Direction;
   int MaxValue;
   int TotalTiles;
   int TileTypes;
   double sum;
   double sumsq;
   double dev;
   double MeanExpectation;
   double SDExpectation;
   char *outfile;
   
   FILE *infp, *outfp;
   
   if(ac < 2){
     Copyright(stdout);
     Usage();
     exit(2);
   }
   
   TilesDesired = DEFAULTTILES;
   MaxValue = DEFMAXVALUE;
   
   while( (opt = getopt(ac,av,"hm:n:v")) != EOF){
      switch(opt){
       case 'n':
	 TilesDesired = atoi(optarg);
	 if(TilesDesired < 2){
	    fprintf(stderr,"Nonsensical number of tiles: %s\n",optarg);
	    exit(1);
	 }
	 break;
       case 'm':
	 MaxValue = atoi(optarg);
	 if(MaxValue < 1){
	    fprintf(stderr,"Nonsensical maximum value: %s\n",optarg);
	    exit(1);
	 }
	 break;
      case 'v':
	Version(stdout);
	exit(2);
      case 'h':
	Usage();
	exit(2);
       default:
	 fprintf(stderr,"Illegal option %c\n",c);
	 exit(1);
      }
   }

   Copyright(stdout);
   
   TilesDesired -= BLANKS;
   
   /* Open input and output files */ 
   infp = OpenFile(av[optind],"r",progname);
   outfile = (char *) malloc (((size_t)(5+strlen(av[optind]))));
   if(outfile == NULL){
     fprintf(stderr,"%s: malloc error.\n",av[0]);
     exit(3);
   }
   sprintf(outfile,"%s.des",basename(av[optind]));
   outfp=OpenFile(outfile,"w",progname);
   
   /* Initialize counts to zero */
   for(i=0;i<CHARCNT;i++) cinfo[i].tokens = 0L;
   TotalChars = 0L;
   

   /* Count characters, using codes as array indices */
   
   printf("Counting characters.\n");
   while((c=getc(infp))!=EOF){
      cinfo[c].tokens+=1;
      ++TotalChars;
   }
   fclose(infp);
   

   /* Compute frequencies */
   
   for(i=0;i<CHARCNT-1;i++){
      cinfo[i].frequency = (double)(cinfo[i].tokens)/(double) TotalChars;
   }
   
   /* Calculate number of tiles */
   
   Iterations = 0;
   Direction = 0;
   TargetTiles = TilesDesired;
   while(TRUE){
      Iterations+=1;
      TotalTiles = 0;
      for(i=0;i<CHARCNT-1;i++){
	 if(cinfo[i].tokens == 0) continue;
	 cinfo[i].ftiles = (double) TargetTiles * cinfo[i].frequency;
	 cinfo[i].tiles = RND(cinfo[i].ftiles);
	 if(cinfo[i].tiles == 0) cinfo[i].tiles = 1;
	 TotalTiles += cinfo[i].tiles;
      }
      printf("Iteration %2d: Target = %3d Total = %3d\n",
	     Iterations,TargetTiles+BLANKS,TotalTiles+BLANKS);
      if(TotalTiles < TilesDesired){
	 if(Direction == -1){
	    fprintf(stderr,"Failed to converge. Printing best result.\n"); 
	    break;
	 }
	 TargetTiles+=1;
	 Direction = 1;
      }
      else if(TotalTiles > TilesDesired){
	 if(Direction == 1){
	    fprintf(stderr,"Failed to converge. Printing best result.\n"); 
	    break;
	 }
	 TargetTiles-=1;
	 Direction = -1;
      }
      else break;
   }
   
   /* Calculate values and expectation */
   
   for(i=0;i<CHARCNT-1;i++){
      if(cinfo[i].frequency < FREQTHRESHOLD) cinfo[i].value = MaxValue;
		else{
			cinfo[i].value = RND((1.0/(double)MaxValue)/cinfo[i].frequency);
			if(cinfo[i].value == 0) cinfo[i].value = 1;
			else if (cinfo[i].value > MaxValue) cinfo[i].value = MaxValue;
		}
      cinfo[i].expectation = cinfo[i].tiles * cinfo[i].value;
   }
   
   /* Evaluate result */
	
   TileTypes = 0;
   sum = 0.0;
   for(i=0;i<CHARCNT-1;i++){
      if(cinfo[i].tokens > 0){
	 TileTypes += 1;
	 sum += (double) cinfo[i].expectation;
      }
   }
   
   MeanExpectation = sum/(double)TileTypes;
   
   sumsq = 0.0;
   dev = 0.0;
   for(i=0;i<CHARCNT-1;i++){
      if(cinfo[i].tokens > 0){
	 dev = cinfo[i].expectation-MeanExpectation;
	 sumsq += (dev * dev);
      }
   }
   SDExpectation = sqrt(sumsq/(double)(TileTypes -1));
   
   fprintf(outfp,"Tile types = %d\n",TileTypes);
   fprintf(outfp,"Total tiles (including blanks) = %d\n",TotalTiles+2);
   fprintf(outfp,"Mean expectation              = %5.2f\n",MeanExpectation);
   fprintf(outfp,"SD of expectation             = %5.2f\n",SDExpectation);
   fprintf(outfp,"Mean deviation of expectation = %5.2f\n",
	   SDExpectation/MeanExpectation);
   
   
   /* Write out the results */
   
   fprintf(outfp,
	   "Character  Tokens   Frequency (%%)  Expectation FTiles Tiles  Value\n\n");
   
   for(i=0;i<CHARCNT-1;i++){
      if(cinfo[i].tokens == 0) continue;
      fprintf(outfp,"   %c       %9ld  %5.2f           %2d       %5.2f  %3d     %2d\n",
	      (char)i,
	      cinfo[i].tokens,
	      cinfo[i].frequency * 100.0,
	      cinfo[i].expectation,
	      cinfo[i].ftiles,
	      cinfo[i].tiles,
	      cinfo[i].value);
   }
   
   fprintf(stdout,"Output written to %s.\n", outfile);
   fclose(outfp);
   exit(0);
}
